import nodes
import unittest
import io

class Test1(unittest.TestCase):

    def setUp(self):
        self.h = nodes.Hierarchy()
        f = io.StringIO("""
nodes: n[1-99]
enclosure e1: n[1-4]
enclosure e2: n[5-8]
enclosure e3: n[9-12]
enclosure e4: n[13-16]
rack r1: e[1-2]
rack r2: e[3-4]
oddball o1: n[6-10]
""")
        self.h.parse_file(f)
        f.close()

    def expands_to(self, nodespec, hostlist):
        self.assertEqual(nodes.collect_name_hostlist(self.h.to_nodes(nodespec)), hostlist)
        
    def expands_bad(self, nodespec):
        self.assertRaises(nodes.NodesException,
                          self.h.to_nodes,  nodespec)

    def test_expand(self):
        self.expands_to("",       "")
        self.expands_to("n1",     "n1")
        self.expands_to("n99",    "n99")
        self.expands_to("e1",     "n[1-4]")
        self.expands_to("e1,e1",  "n[1-4]")
        self.expands_to("e[1-2]", "n[1-8]")
        self.expands_to("e[1,3]", "n[1-4,9-12]")
        self.expands_to("e1,n1",  "n[1-4]")
        self.expands_to("e1,n5",  "n[1-5]")
        self.expands_to("n6,e1",  "n[1-4,6]")
        self.expands_to("r1",     "n[1-8]")
        self.expands_to("r1,e3",  "n[1-12]")
        self.expands_to("r1,e4",  "n[1-8,13-16]")
        self.expands_to("o1",     "n[6-10]")
        self.expands_to("o1,e1",  "n[1-4,6-10]")
        self.expands_to("o1,e2",  "n[5-10]")
        self.expands_to("o1,e3",  "n[6-12]")
        
        self.expands_bad("n0")
        self.expands_bad("n100")

    def ufrg(self, level, nodespec, up, fill, representative, gather):
        if up is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.up(level, nodespec)), up)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.up, level, nodespec)

        if fill is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.fill(level, nodespec)), fill)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.fill, level, nodespec)

        if representative is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.representative(level, nodespec)), representative)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.representative, level, nodespec)

        if gather is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.gather(level, nodespec)), gather)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.gather, level, nodespec)

    def ufrg_e(self, nodespec, up, fill, representative, gather):
        self.ufrg("enclosure", nodespec, up, fill, representative, gather)

    def ufrg_r(self, nodespec, up, fill, representative, gather):
        self.ufrg("rack", nodespec, up, fill, representative, gather)

    def test_up_fill_representative_gather(self):
        #           NODESPEC     UP          FILL            REPRESENTATIVE  GATHER
        # -----------------------------------------------------------------------------------
        self.ufrg_e("",          "",         "",             "",             "")
        self.ufrg_e("e1",        "e1",       "n[1-4]",       "n1",           "e1")  
        self.ufrg_e("n1",        "e1",       "n[1-4]",       "n1",           "n1")
        self.ufrg_e("n2",        "e1",       "n[1-4]",       "n1",           "n2")
        self.ufrg_e("n3",        "e1",       "n[1-4]",       "n1",           "n3")
        self.ufrg_e("n4",        "e1",       "n[1-4]",       "n1",           "n4")
        self.ufrg_e("n5",        "e2",       "n[5-8]",       "n5",           "n5")
        self.ufrg_e("n[1-2]",    "e1",       "n[1-4]",       "n1",           "n[1-2]")
        self.ufrg_e("n[4-5]",    "e[1-2]",   "n[1-8]",       "n[1,5]",       "n[4-5]")
        self.ufrg_e("n[1-5,14]", "e[1-2,4]", "n[1-8,13-16]", "n[1,5,13]",    "e1,n[5,14]")
        self.ufrg_e("n[1-8,14]", "e[1-2,4]", "n[1-8,13-16]", "n[1,5,13]",    "e[1-2],n14")
        self.ufrg_e("n[1-9,14]", "e[1-4]",   "n[1-16]",      "n[1,5,9,13]",  "e[1-2],n[9,14]")
                                                                            
        self.ufrg_e("r1",        "e[1-2]",   "n[1-8]",       "n[1,5]",       "e[1-2]")
        self.ufrg_e("r1,e3",     "e[1-3]",   "n[1-12]",      "n[1,5,9]",     "e[1-3]")
        self.ufrg_e("r1,e4",     "e[1-2,4]", "n[1-8,13-16]", "n[1,5,13]",    "e[1-2,4]")
        self.ufrg_e("r1,n16",    "e[1-2,4]", "n[1-8,13-16]", "n[1,5,13]",    "e[1-2],n16")
        self.ufrg_e("r[1-2]",    "e[1-4]",   "n[1-16]",      "n[1,5,9,13]",  "e[1-4]")
                                                                            
        self.ufrg_e("o1",        "e[2-3]",   "n[5-12]",      "n[5,9]",       "n[6-10]")
                                                                            
        self.ufrg_e("e5",        None,       None,           None,           None)
        self.ufrg_e("n17",       None,       "n17",          "n17",          "n17")
        self.ufrg_e("n[16-17]",  None,       "n[13-17]",     "n[13,17]",     "n[16-17]") # representative?
                                                                            
        self.ufrg_r("r1",        "r1",       "n[1-8]",       "n1",           "r1")
        self.ufrg_r("e1",        "r1",       "n[1-8]",       "n1",           "n[1-4]")
        self.ufrg_r("n1",        "r1",       "n[1-8]",       "n1",           "n1")
        self.ufrg_r("e[1-2]",    "r1",       "n[1-8]",       "n1",           "r1")
        self.ufrg_r("e[1-3]",    "r[1-2]",   "n[1-16]",      "n[1,9]",       "n[9-12],r1")
        self.ufrg_r("e[1-4]",    "r[1-2]",   "n[1-16]",      "n[1,9]",       "r[1-2]")
        self.ufrg_r("r1,e4",     "r[1-2]",   "n[1-16]",      "n[1,9]",       "n[13-16],r1")
        self.ufrg_r("r1,n16",    "r[1-2]",   "n[1-16]",      "n[1,9]",       "n16,r1")
                                                                            
        self.ufrg_r("o1",        "r[1-2]",   "n[1-16]",      "n[1,9]",       "n[6-10]")
                                                                            
        self.ufrg_r("r3",        None,       None,           None,           None)
        self.ufrg_r("n17",       None,       "n17",          "n17",          "n17")
        self.ufrg_r("n[16-17]",  None,       "n[9-17]",      "n[9,17]",      "n[16-17]") # representative?

    def um(self, level, nodespec, up, up_missing):
        if up is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.up(level, nodespec)), up)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.up, level, nodespec)
        if up_missing is not None:
            self.assertEqual(nodes.collect_name_hostlist(self.h.up(level, nodespec, "M")), up_missing)
        else:
            self.assertRaises(nodes.NodesException,
                              self.h.up, level, nodespec, "M")

    def um_e(self, nodespec, up, up_missing):
        self.um("enclosure", nodespec, up, up_missing)

    def test_up_missing(self):
        #         NODESPEC     UP          UP_MISSING
        # ---------------------------------------------
        self.um_e("n1",        "e1",       "e1")
        self.um_e("n17",       None,       "M")
        self.um_e("n[16-17]",  None,       "M,e4")
        self.um_e("n[1-20]",   None,       "M,e[1-4]")


if __name__ == '__main__':
    unittest.main()

        
